# This build script must be executed from a 'Developer PowerShell for VS 2022' session

# Folder under 'C:\Program Files (x86)\Windows Kits\10\bin' to reference
$windows_sdk_version = "10.0.26100.0"

$wdk_assembly_version = "10.0.26100.4202"

# Folder under 'C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Tools\MSVC' to reference
$msvc_version = "14.44.35207"

# Folder under 'C:\Program Files (x86)\Windows Kits\NETFXSDK' to reference
$netfx_sdk_version = "4.8.1"

# KMDF version to use
$kmdf_version = "1.15"

# Build output directory
$build_dir = "build"

# Name prefix for the driver output files
$driver_name = "my_driver"

# -----------------------------------------------------------------------------

$ErrorActionPreference = "Stop"

function Exit-OnFail {
    param (
        [int]$StatusCode = 0,
        [string]$Desc
    )

    if ($LASTEXITCODE -ne $StatusCode) {
        throw ("Failed to execute '$Desc' command: status code was $LASTEXITCODE, wanted $StatusCode")
    }
}

$src_files = Get-ChildItem -Path src -File | %{ $_.Name }
$qualified_src_files = $src_files | %{ "src\$_" }
$qualified_obj_files = $src_files | %{ $build_dir + "\" + $_.Split(".", 2)[0] + ".obj" }

New-Item -ItemType Directory -Path $build_dir -Force | Out-Null

# check if there is a cert already, and if not, generate one
if (Get-ChildItem -Path Cert:\CurrentUser\My | ?{ $_.Subject -eq 'CN=driver_research' }) {
    Write-Host "Reusing existing 'driver_research' cert"
} else {
    Write-Host "Generating self-signed certificate"
    # https://learn.microsoft.com/en-us/windows/msix/package/create-certificate-package-signing#use-new-selfsignedcertificate-to-create-a-certificate
    New-SelfSignedCertificate -Type Custom -KeyUsage DigitalSignature -CertStoreLocation "Cert:\CurrentUser\My" -TextExtension @("2.5.29.37={text}1.3.6.1.5.5.7.3.3", "2.5.29.19={text}") -Subject "CN=driver_research"
}
    
$cert = (Get-ChildItem -Path Cert:\CurrentUser\My | ?{ $_.Subject -eq 'CN=driver_research' })[0]

# if the cert isn't already in the build folder, re-fetch it
$cert_path = "$build_dir\$driver_name.cer"
if (-not (Test-Path $cert_path)) {
    Write-Host "Exporting certificate to build directory"
    $cert = (Get-ChildItem -Path Cert:\CurrentUser\My | ?{ $_.Subject -eq 'CN=driver_research' })[0]
    Export-Certificate -Cert $cert -FilePath $cert_path | Out-Null
}

Write-Host "Stamping .inf file"
Copy-Item driver.inf $build_dir\$driver_name.inf
& stampinf.exe -d * -a amd64 -v * -k $kmdf_version -x -f "$build_dir\$driver_name.inf" | Out-Null
Exit-OnFail -Desc stampinf

Write-Host "Generating wpp trace files"
& tracewpp.exe "-cfgdir:C:\Program Files (x86)\Windows Kits\10\bin\$windows_sdk_version\wppconfig\rev1" -scan:inc\trace.h -odir:$build_dir\ -km $qualified_src_files
Exit-OnFail -Desc tracewpp

Write-Host "Compiling objects"
$env:CAExcludePath = "C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Tools\MSVC\$msvc_version\atlmfc\include;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Tools\MSVC\$msvc_version\include;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\ucrt;;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\um;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\shared;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\winrt;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\cppwinrt;C:\Program Files (x86)\Windows Kits\NETFXSDK\$netfx_sdk_version\Include\um;;C:\Program Files (x86)\Windows Kits\10\Include"
$env:EXTERNAL_INCLUDE = "C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Tools\MSVC\$msvc_version\include;;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Tools\MSVC\$msvc_version\atlmfc\include;;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Auxiliary\VS\include;;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\ucrt;;;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Auxiliary\VS\UnitTest\include;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\um;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\shared;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\winrt;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\cppwinrt;C:\Program Files (x86)\Windows Kits\NETFXSDK\$netfx_sdk_version\Include\um;"
$env:INCLUDE = "C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\km\crt;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\km;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\shared;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\shared\;C:\Program Files (x86)\Windows Kits\10\Include\wdf\kmdf\$kmdf_version;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Tools\MSVC\$msvc_version\include;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Tools\MSVC\$msvc_version\atlmfc\include;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Auxiliary\VS\include;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\ucrt;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Auxiliary\VS\UnitTest\include;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\um;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\winrt;C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\cppwinrt;C:\Program Files (x86)\Windows Kits\NETFXSDK\$netfx_sdk_version\Include\um"
$env:LIB = "C:\Program Files (x86)\Windows Kits\10\lib\$windows_sdk_version\km\x64;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Tools\MSVC\$msvc_version\lib\spectre\x64;;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Tools\MSVC\$msvc_version\atlmfc\lib\spectre\x64;;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Auxiliary\VS\lib\x64;;C:\Program Files (x86)\Windows Kits\10\lib\$windows_sdk_version\ucrt\x64;;;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Auxiliary\VS\UnitTest\lib;C:\Program Files (x86)\Windows Kits\10\lib\$windows_sdk_version\um\x64;C:\Program Files (x86)\Windows Kits\NETFXSDK\$netfx_sdk_version\lib\um\x64;"
$env:LIBPATH = "C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Tools\MSVC\$msvc_version\atlmfc\lib\spectre\x64;;C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Tools\MSVC\$msvc_version\lib\spectre\x64;"
& cl.exe /ifcOutput $build_dir\ /GS /W4 /wd"4748" /wd"4603" /wd"4627" /wd"4986" /wd"4987" /Gy /Zc:wchar_t- /I"$build_dir\" /I"inc\" /analyze:"stacksize1024" /guard:cf /Zi /Gm- /Od /Fd"$build_dir\vc143.pdb" /FI"C:\Program Files (x86)\Windows Kits\10\Include\$windows_sdk_version\shared\warning.h" /Zc:inline /fp:precise /Zp8 /D "_WIN64" /D "_AMD64_" /D "AMD64" /D "DEPRECATE_DDK_FUNCTIONS=1" /D "MSC_NOOPT" /D "_WIN32_WINNT=0x0A00" /D "WINVER=0x0A00" /D "WINNT=1" /D "NTDDI_VERSION=0xA000010" /D "DBG=1" /GF /WX /Zc:forScope /GR- /Gz /Oy- /Oi /FC /Fa"$build_dir\" /nologo /Fo"$build_dir\" /Fp"$build_dir\driver.pch" /diagnostics:column /c $qualified_src_files
Exit-OnFail -Desc cl

Write-Host "Linking objects"
& link.exe /OUT:"$build_dir\$driver_name.sys" /MANIFEST:NO /PROFILE /Driver /PDB:"$build_dir\$driver_name.pdb" "C:\Program Files (x86)\Windows Kits\10\lib\$windows_sdk_version\km\x64\BufferOverflowFastFailK.lib" "C:\Program Files (x86)\Windows Kits\10\lib\$windows_sdk_version\km\x64\ntoskrnl.lib" "C:\Program Files (x86)\Windows Kits\10\lib\$windows_sdk_version\km\x64\hal.lib" "C:\Program Files (x86)\Windows Kits\10\lib\$windows_sdk_version\km\x64\wmilib.lib" "C:\Program Files (x86)\Windows Kits\10\lib\wdf\kmdf\x64\$kmdf_version\WdfLdr.lib" "C:\Program Files (x86)\Windows Kits\10\lib\wdf\kmdf\x64\$kmdf_version\WdfDriverEntry.lib" /RELEASE /VERSION:"10.0" /DEBUG /MACHINE:X64 /ENTRY:"FxDriverEntry" /WX /OPT:REF /INCREMENTAL:NO /SUBSYSTEM:NATIVE",10.00" /OPT:ICF /MERGE:"_TEXT=.text;_PAGE=PAGE" /ILK:"$build_dir\$driver_name.ilk" /NOLOGO /NODEFAULTLIB /SECTION:"INIT,d" $qualified_obj_files
Exit-OnFail -Desc link

Write-Host "Signing driver"
& signtool.exe sign /ph /fd sha256 /sha1 $cert.Thumbprint $build_dir\$driver_name.sys
Exit-OnFail -Desc "signtool .sys"

Write-Host "Generating catalog file"
New-Item -ItemType Directory -Path $build_dir -Force | Out-Null

& "C:\Program Files (x86)\Windows Kits\10\bin\$windows_sdk_version\x86\inf2cat.exe" /os:10_x64 "/driver:$build_dir"
Exit-OnFail -Desc inf2cat

& drvcat.exe "$build_dir\$driver_name.cat" /set WdkVersion $windows_sdk_version /set WdkAssemblyVersion $wdk_assembly_version
Exit-OnFail -Desc drvcat

Write-Host "Signing catalog file"
& signtool.exe sign /ph /fd sha256 /sha1 $cert.Thumbprint $build_dir\$driver_name.cat
Exit-OnFail -Desc "signtool .cat"